/** @file   highscoresmenu.cpp
 * @brief   Implementation of HighscoresMenu - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "highscoresmenu.h"
#include "GfxManager.h"
#include "highscoretable.h"
#include "consts.h"
#include "soundsamples.h"

using namespace eng2d;
using std::string;
using std::vector;

namespace WeWantWar {

/** Constructor
 */
HighscoresMenu::HighscoresMenu() :
  MenuPage()
{
  this->createGraphics();
}



/** Destructor
 */
HighscoresMenu::~HighscoresMenu()
{
  this->cleanupGraphics();
}


/** The update method
 */
int HighscoresMenu::update()
{
  int ret = MenuPage::NO_PAGE_CHANGE;
  
  if ( this->getState() == MenuPage::STATE_FADEIN ) {
    ret = this->updateFadeInState();
    
  } else if ( this->getState() == MenuPage::STATE_VISIBLE ) {
    ret = this->updateVisibleState();
    
  } else if ( this->getState() == MenuPage::STATE_FADEOUT ) {
    ret = this->updateFadeOutState();
    
  }
  
  return ret;
}



/** Redraws this menu
 */
void HighscoresMenu::redraw()
{
  blit( GfxManager::titlePic, Display::buffer, 0,0, 0,0, 420,480 );
  
  // Draw the highscore sprites
  for ( int i=0; i < m_sprites.size(); i++ ) {
    m_sprites.at(i)->draw( Display::buffer );
  }
}


/** Returns the pageid
 */
MenuPage::PageID HighscoresMenu::getPageID() const
{
  return MenuPage::HIGHSCORES;
}


/** Creates the sprites
 */
void HighscoresMenu::createGraphics()
{
  HighscoreTable hscore( Consts::HIGHSCOREFILE_NAME );
  int y = 490;
  for ( int i=0; i < hscore.numberOfEntries(); i++ ) {
    BITMAP* pB = create_bitmap( 300,24 );
    HighscoreTable::Entry e = hscore.getEntry( i );
    this->drawEntry( pB, e.scores, e.name );
    Sprite* pS = new Sprite( pB );
    pS->position( Vec2D( 60, y ) );
    m_sprites.push_back( pS );
    y += 25;
  }
}

/** Draws the highscore entry to the bitmap
 */
void HighscoresMenu::drawEntry( BITMAP* pB, int score, const string& name )
{
  if ( pB == 0 ) {
    return;
  }
  
  clear_to_color( pB, makecol( 255,0,255 ) );
  TPrinter out( pB );
  out.font( GfxManager::titleFont, "menuitems" );
  out.pos( 0,0 );
  out.print( name );
  
  out.align( TPrinter::ALIGN_RIGHT );
  out.pos( 290,0 );
  out.font( GfxManager::titleFont, "menuitems_sel" );
  out.print( score );
}

/** Destroyes the graphics
 */
void HighscoresMenu::cleanupGraphics()
{
  for ( int i=0; i < m_sprites.size(); i++ ) {
    if ( m_sprites.at(i) != 0 ) {
      if ( m_sprites.at(i)->al_BITMAP() != 0 ) {
        destroy_bitmap( m_sprites.at(i)->al_BITMAP() );
      }
      delete m_sprites.at(i);
      m_sprites.at(i) = 0;
    }
  }
  m_sprites.clear();
}

/** Updates the fadein state
 */
int HighscoresMenu::updateFadeInState()
{
  for ( int i=0; i < m_sprites.size(); i++ ) {
    Vec2D pos( m_sprites.at(i)->position() );
    float ny = pos.y() - 20;
    pos.y( ny );
    m_sprites.at(i)->position( pos );
  }
  
  Vec2D pos( m_sprites.at(0)->position() );
  if ( pos.y() < 230 ) {
    MenuPage::m_state = MenuPage::STATE_VISIBLE;
  }
  return MenuPage::NO_PAGE_CHANGE;
}

/** Updates the visible state
 */
int HighscoresMenu::updateVisibleState()
{
  if ( key[KEY_ENTER] || key[KEY_SPACE] ) {
    MenuPage::m_state = MenuPage::STATE_FADEOUT;
    Sound::playSample( SMP_MENUSELECT, false );
  }
  return MenuPage::NO_PAGE_CHANGE;
}

/** Updates the fadeout state
 */
int HighscoresMenu::updateFadeOutState()
{
  for ( int i=0; i < m_sprites.size(); i++ ) {
    Vec2D pos( m_sprites.at(i)->position() );
    float ny = pos.y() + 20;
    pos.y( ny );
    m_sprites.at(i)->position( pos );
  }

  Vec2D pos( m_sprites.at(0)->position() );
  if ( pos.y() > 480 ) {
    return MenuPage::PAGE_CHANGE;
  }
  return MenuPage::NO_PAGE_CHANGE;
}


} // end of namespace
